<?php

namespace App\Http\Controllers;

use App\Blog;
use App\BlogMenu;
use App\Menu;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Intervention\Image\Facades\Image;
use App\FetauredBlog;
use Illuminate\Support\Facades\Log;



class BlogController extends Controller
{
   

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function __construct()
    {
        $this->middleware('permission:blog.view', ['only' => ['index']]);
        $this->middleware('permission:blog.create', ['only' => ['create', 'store']]);
        $this->middleware('permission:blog.edit', ['only' => ['edit', 'update', 'status_update']]);
        $this->middleware('permission:blog.delete', ['only' => ['destroy', 'bulk_delete']]);
    }

    public function index(Request $request)
    {
        if ($request->search != null) {
            $blogs = Blog::where('title', 'like', '%' . $request->search . '%')->select('id', 'title', 'image', 'is_active', 'detail', 'created_at', 'updated_at')->orderBy('id', 'DESC')->paginate(12);
        } else {
            $blogs = Blog::select('id', 'title', 'image', 'is_active', 'detail', 'created_at', 'updated_at')->orderBy('id', 'DESC')->paginate(12);
        }

        return view('admin.blog.index', compact('blogs'));
    }

    public function create()
    {
        $menus = Menu::all();
        return view('admin.blog.create', compact('menus'));
    }

    public function store(Request $request)
    {
        if (config('app.demolock') == 1) {
            return back()->with('deleted', __('This action is disabled in the demo !'));
        }
        $request->validate([
            'title' => 'required',
            'detail' => 'required',
        ], [
            'menu.required' => __('Please select atleast one menu'),
        ]);

        $input = $request->all();
        $input['detail'] = $request->detail;

        if ($file = $request->file('image')) {
            $validator = Validator::make(
                [
                    'image' => $request->image,
                    'extension' => strtolower($request->image->getClientOriginalExtension()),
                ],
                [
                    'image' => 'required',
                    'image' => 'required|file|image|mimes:jpeg,png,jpg,webp',
                ]
            );
            if ($validator->fails()) {
                return back()->with('deleted', __('Invalid file format Please use jpg,webp,jpeg and png image format !'))->withInput();
            } else {

                $thumbnail = 'thumb_' . time() . $file->getClientOriginalName();
                $img = Image::make($file->path());
                $img->resize(300, 450, function ($constraint) {
                    $constraint->aspectRatio();
                });
                $img->save(public_path('/images/blog') . '/' . $thumbnail);

                $input['image'] = $thumbnail;
            }

        }

        if ($file = $request->file('poster')) {
            $validator = Validator::make(
                [
                    'poster' => $request->poster,
                    'extension' => strtolower($request->poster->getClientOriginalExtension()),
                ],
                [
                    'poster' => 'required',
                    'poster' => 'required|file|image|mimes:jpeg,png,jpg,webp',
                ]
            );
            if ($validator->fails()) {
                return back()->with('deleted', __('Invalid file format Please use jpg,webp,jpeg and png image format !'))->withInput();
            } else {
                $poster = 'poster_' . time() . $file->getClientOriginalName();
                $img = Image::make($file->path());

                $img->resize(300, 169, function ($constraint) {
                    $constraint->aspectRatio();
                });
                $img->save(public_path('/images/blog') . '/' . $poster);

                $input['poster'] = $poster;
            }

        }

        if (isset($input['is_active']) && $input['is_active'] == '1') {
            $input['is_active'] = 1;
        } else {
            $input['is_active'] = 0;
        }

        $slug = str_slug($input['title'], '-');
        $input['slug'] = $slug;
        $auth = Auth::user()->id;
        $input['user_id'] = $auth;

        $menus = null;

        try {
            $blog = Blog::create($input);

            

            return back()->with('success', __('Post has been added'));
        } catch (\Exception $e) {

            return back()->with('deleted', $e->getMessage())->withInput();
        }

    }

    public function showBlogList()
{
    $auth = Auth::user();
    $blogs = Blog::orderBy('created_at', 'desc')->where('is_active', '1')->get();
    $featuredBlog = FetauredBlog::first();

    // Check if $featuredBlog is not null and has the 'blog_names' property
    if ($featuredBlog && isset($featuredBlog->blog_names)) {
        $blogIds = json_decode($featuredBlog->blog_names, true); // Decode JSON to array
        $featuredBlogs = DB::table('blogs')->whereIn('id', $blogIds)->get();
    } else {
        $featuredBlogs = collect(); // Return an empty collection if no featured blogs are found
    }

    return view('blog', compact('blogs', 'auth', 'featuredBlogs', 'featuredBlog'));
}


    public function showBlog($slug)
    {
        $blogdetail = Blog::where('slug', $slug)->first();
        $exceptblog = Blog::where('slug', '!=', $slug)->orderBy('id','DESC')->get();
        return view('blogdetail', compact('blogdetail', 'exceptblog'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Coupon  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $blog = Blog::findOrFail($id);
        $menus = Menu::all();
        return view('admin.blog.edit', compact('blog', 'menus'));

    }

/**
 * Update the specified resource in storage.
 *
 * @param  \Illuminate\Http\Request  $request
 * @param  \App\Product  $id
 * @return \Illuminate\Http\Response
 */

    public function update(Request $request, $id)
    {
        if (config('app.demolock') == 1) {
            return back()->with('deleted', __('This action is disabled in the demo !'));
        }

        $request->validate([
            'title' => 'required|min:3|unique:blogs,title,' . $id,
            'detail' => 'required|min:3',
        ]);

        $blog = Blog::findOrFail($id);
        $input = $request->all();
        $input['detail'] = $request->detail;
        foreach ($blog->blog_m as $key => $bm) {
            # code...
            $bm->delete();
        }

        if ($file = $request->file('image')) {
            $validator = Validator::make(
                [
                    'image' => $request->image,
                    'extension' => strtolower($request->image->getClientOriginalExtension()),
                ],
                [
                    'image' => 'required',
                    'image' => 'required|file|image|mimes:jpeg,png,jpg,webp',
                ]
            );
            if ($validator->fails()) {
                return back()->with('deleted', __('Invalid file format Please use jpg,webp,jpeg and png image format !'))->withInput();
            } else {

                if ($blog->image != null) {

                    $image_file = @file_get_contents(public_path() . '/images/blog/' . $blog->image);

                    if ($image_file) {
                        unlink(public_path() . '/images/blog/' . $blog->image);
                    }

                }

                $thumbnail = 'thumb_' . time() . $file->getClientOriginalName();
                $img = Image::make($file->path());

                $img->resize(300, 450, function ($constraint) {
                    $constraint->aspectRatio();
                });
                $img->save(public_path('/images/blog') . '/' . $thumbnail);
                $input['image'] = $thumbnail;
            }

        }

        if ($file = $request->file('poster')) {
            $validator = Validator::make(
                [
                    'poster' => $request->poster,
                    'extension' => strtolower($request->poster->getClientOriginalExtension()),
                ],
                [
                    'poster' => 'required',
                    'poster' => 'required|file|image|mimes:jpeg,png,jpg,webp',
                ]
            );
            if ($validator->fails()) {
                return back()->with('deleted', __('Invalid file format Please use jpg,webp,jpeg and png image format !'))->withInput();
            } else {

                if ($blog->poster != null) {

                    $image_file = @file_get_contents(public_path() . '/images/blog' . $blog->poster);

                    if ($image_file) {
                        unlink(public_path() . '/images/blog' . $blog->image);
                    }

                }

                $poster = 'poster_' . time() . $file->getClientOriginalName();
                $img = Image::make($file->path());

                $img->resize(300, 169, function ($constraint) {
                    $constraint->aspectRatio();
                });
                $img->save(public_path('/images/blog') . '/' . $poster);

                $input['poster'] = $poster;
            }

        }

        if (isset($request->is_active)) {
            $input['is_active'] = '1';
        } else {

            $input['is_active'] = '0';
        }

        $slug = str_slug($input['title'], '-');

        $input['slug'] = $slug;

        try {
            $blog->update($input);
            

            return redirect('admin/blog')->with('success', __('Post has been updated'));
        } catch (\Exception $e) {

            return back()->with('deleted', $e->getMessage())->withInput();
        }
    }

    public function destroy($id)
    {
        if (config('app.demolock') == 1) {
            return back()->with('deleted', __('This action is disabled in the demo !'));
        }
        $blog = Blog::findOrFail($id);
        if ($blog->image != null) {
            $content = @file_get_contents(public_path() . '/images/blog' . $blog->image);
            if ($content) {
                unlink(public_path() . "/images/blog" . $blog->image);
            }
        }
        if ($blog->poster != null) {
            $content = @file_get_contents(public_path() . '/images/blog' . $blog->poster);
            if ($content) {
                unlink(public_path() . "/images/blog" . $blog->poster);
            }
        }
        $blog_menu = BlogMenu::where('blog_id', $id)->delete();
        $blog->delete();

        return back()->with('deleted', __('Post has been deleted'));
    }

public function bulk_delete(Request $request)
{
    if (config('app.demolock') == 1) {
        return back()->with('deleted', __('This action is disabled in the demo !'));
    }

    $validator = Validator::make($request->all(), [
        'checked' => 'required|array',
        'checked.*' => 'integer|exists:blogs,id'
    ]);

    if ($validator->fails()) {
        return back()->with('deleted', __('Please select one or more items to delete'));
    }

    try {
        foreach ($request->checked as $blogId) {
            $blog = Blog::findOrFail($blogId);
            
            // Delete image if exists
            if ($blog->image != null) {
                $imagePath = public_path('/images/blog/' . $blog->image);
                if (file_exists($imagePath)) {
                    unlink($imagePath);
                }
            }
            
            // Delete poster if exists
            if ($blog->poster != null) {
                $posterPath = public_path('/images/blog/' . $blog->poster);
                if (file_exists($posterPath)) {
                    unlink($posterPath);
                }
            }
            
            // Delete related blog menu entries
            BlogMenu::where('blog_id', $blogId)->delete();
            
            // Delete the blog
            $blog->delete();
        }
        
        return back()->with('deleted', __('Selected posts have been deleted successfully'));
        
    } catch (\Exception $e) {
        return back()->with('error', __('An error occurred while deleting posts: ') . $e->getMessage());
    }
}

}
